<?php

// file: user_ledger.php - User Dashboard (Redesigned)
require_once "config.php";
require_once "link_crypto.php";

// Get and decrypt token
$token = $_GET['token'] ?? '';
if (empty($token)) {
    die("❌ Missing security token. Please request a new link from the bot.");
}

$payload = decrypt_secure_token($token);
if (!$payload) {
    die("❌ Invalid or corrupted token. Please request a new link from the bot.");
}

// Check token expiration (24 hours)
if (is_token_expired($payload['created'], 24)) {
    die("❌ This link has expired. Links are valid for 24 hours only. Please request a new link from the bot.");
}


// Extract parameters from token
$uid = $payload['uid'];
$gid = $payload['gid'];

$start_date = $payload['start_date'];
$end_date = $payload['end_date'];

// Restrict to current day only
if (!is_current_day($start_date)) {
    die("❌ 此链接仅适用于当日数据 (This link is only valid for current day data). Please request a new link from the bot.");
}

// Search filter
$search_query = $_GET['search'] ?? '';
$search_by = $_GET['search_by'] ?? 'marker'; // marker, operator, remark

// Format dates for display and SQL
$start_datetime = $start_date;
$end_datetime = $end_date;

// Get settings (exchange rate, fee)
$settings_stmt = $conn->prepare("SELECT exchange_rate, fee_rate FROM settings WHERE group_id = $gid");
$settings_stmt->execute();
$settings_res = $settings_stmt->get_result();
$settings = $settings_res->fetch_assoc() ?: ['exchange_rate' => 1.0, 'fee_rate' => 0.0];
$settings_stmt->close();

$global_exchange_rate = (float)$settings['exchange_rate'];
$global_fee_rate = (float)$settings['fee_rate'];

// Get user info
$user_info_stmt = $conn->prepare("SELECT telegram_username FROM transactions WHERE telegram_user_id = ? LIMIT 1");
$user_info_stmt->bind_param("i", $uid);
$user_info_stmt->execute();
$user_info_res = $user_info_stmt->get_result();
$user_info = $user_info_res->fetch_assoc();


$username = $user_info['telegram_username'] ?? "User #$uid";
$user_info_stmt->close();

// Base query with search filter
$search_condition = "";
$search_params = [];
$search_types = "";

if (!empty($search_query)) {
    if ($search_by === 'marker') {
        $search_condition = " AND telegram_username LIKE ?";
        $search_params[] = "%$search_query%";
        $search_types = "s";
    } elseif ($search_by === 'operator') {
        $search_condition = " AND operator_name LIKE ?";
        $search_params[] = "%$search_query%";
        $search_types = "s";
    } elseif ($search_by === 'remark') {
        $search_condition = " AND remark LIKE ?";
        $search_params[] = "%$search_query%";
        $search_types = "s";
    }
}

// =========================
// 1. DEPOSITS (IN transactions)
// =========================
$deposits_query = "
    SELECT created_at, amount, telegram_username as marker, operator_name, remark, transaction_exchange_rate
    FROM transactions 
    WHERE telegram_user_id = ? 
    AND group_id = ? 
    AND type = 'IN'
    AND created_at BETWEEN ? AND ?
    $search_condition
    ORDER BY created_at ASC
";
$deposits_stmt = $conn->prepare($deposits_query);
if (!empty($search_params)) {
    $deposits_stmt->bind_param("isss" . $search_types, $uid, $gid, $start_datetime, $end_datetime, ...$search_params);
} else {
    $deposits_stmt->bind_param("isss", $uid, $gid, $start_datetime, $end_datetime);
}
$deposits_stmt->execute();
$deposits_result = $deposits_stmt->get_result();

// echo "SQL:\n";
// echo $deposits_query . "\n\n";

// echo "Parameters:\n";
// $params = [$uid, $start_datetime, $end_datetime];
// if (!empty($search_params)) {
//     $params = array_merge($params, $search_params);
// }
// print_r($params);
// exit;

$deposits = [];
$total_deposits = 0;
while ($row = $deposits_result->fetch_assoc()) {
    $deposits[] = $row;
    $total_deposits += (float)$row['amount'];
}
$deposits_stmt->close();

// =========================
// 2. ISSUED (OUT transactions)
// =========================
$issued_query = "
    SELECT created_at, amount, telegram_username as marker, operator_name, remark, transaction_exchange_rate
    FROM transactions 
    WHERE telegram_user_id = ? 
    AND group_id = ?
    AND type = 'OUT'
    AND created_at BETWEEN ? AND ?
    $search_condition
    ORDER BY created_at ASC
";
$issued_stmt = $conn->prepare($issued_query);
if (!empty($search_params)) {
    $issued_stmt->bind_param("isss" . $search_types, $uid, $gid, $start_datetime, $end_datetime, ...$search_params);
} else {
    $issued_stmt->bind_param("isss", $uid, $gid, $start_datetime, $end_datetime);
}
$issued_stmt->execute();
$issued_result = $issued_stmt->get_result();
$issued = [];
$total_issued = 0;
while ($row = $issued_result->fetch_assoc()) {
    $issued[] = $row;
    $total_issued += (float)$row['amount'];
}
$issued_stmt->close();

// =========================
// 3. STATISTICS BY MARKER (username)
// =========================
$stats_marker_query = "
    SELECT 
        telegram_username as username,
        SUM(CASE WHEN type='IN' THEN amount ELSE 0 END) as deposit,
        SUM(CASE WHEN type='OUT' THEN amount ELSE 0 END) as already_issued,
        SUM(CASE WHEN type='IN' THEN amount ELSE 0 END) - SUM(CASE WHEN type='OUT' THEN amount ELSE 0 END) as not_issued
    FROM transactions
    WHERE telegram_user_id = ?
    AND group_id = ?
    AND created_at BETWEEN ? AND ?
    $search_condition
    GROUP BY telegram_username
    HAVING deposit != 0 OR already_issued != 0
";
$stats_marker_stmt = $conn->prepare($stats_marker_query);
if (!empty($search_params)) {
    $stats_marker_stmt->bind_param("isss" . $search_types, $uid, $gid, $start_datetime, $end_datetime, ...$search_params);
} else {
    $stats_marker_stmt->bind_param("isss", $uid, $gid, $start_datetime, $end_datetime);
}
$stats_marker_stmt->execute();
$stats_marker_result = $stats_marker_stmt->get_result();
$stats_by_marker = [];
while ($row = $stats_marker_result->fetch_assoc()) {
    $stats_by_marker[] = $row;
}
$stats_marker_stmt->close();

// =========================
// 4. STATISTICS BY OPERATOR
// =========================
$stats_operator_query = "
    SELECT 
        operator_name as username,
        SUM(CASE WHEN type='IN' THEN amount ELSE 0 END) as deposit,
        SUM(CASE WHEN type='OUT' THEN amount ELSE 0 END) as already_issued,
        SUM(CASE WHEN type='IN' THEN amount ELSE 0 END) - SUM(CASE WHEN type='OUT' THEN amount ELSE 0 END) as not_issued
    FROM transactions
    WHERE telegram_user_id = ?
    AND group_id = ?
    AND created_at BETWEEN ? AND ?
    $search_condition
    GROUP BY operator_name
    HAVING deposit != 0 OR already_issued != 0
";
$stats_operator_stmt = $conn->prepare($stats_operator_query);
if (!empty($search_params)) {
    $stats_operator_stmt->bind_param("isss" . $search_types, $uid, $gid, $start_datetime, $end_datetime, ...$search_params);
} else {
    $stats_operator_stmt->bind_param("isss", $uid, $gid, $start_datetime, $end_datetime);
}
$stats_operator_stmt->execute();
$stats_operator_result = $stats_operator_stmt->get_result();
$stats_by_operator = [];
while ($row = $stats_operator_result->fetch_assoc()) {
    $stats_by_operator[] = $row;
}
$stats_operator_stmt->close();

// =========================
// 5. STATISTICS BY NOTES (remark)
// =========================
$stats_notes_query = "
    SELECT 
        remark as username,
        SUM(CASE WHEN type='IN' THEN amount ELSE 0 END) as deposit,
        SUM(CASE WHEN type='OUT' THEN amount ELSE 0 END) as already_issued,
        SUM(CASE WHEN type='IN' THEN amount ELSE 0 END) - SUM(CASE WHEN type='OUT' THEN amount ELSE 0 END) as balance
    FROM transactions
    WHERE telegram_user_id = ?
    AND group_id = ?
    AND created_at BETWEEN ? AND ?
    $search_condition
    GROUP BY remark
    HAVING deposit != 0 OR already_issued != 0
";
$stats_notes_stmt = $conn->prepare($stats_notes_query);
if (!empty($search_params)) {
    $stats_notes_stmt->bind_param("isss" . $search_types, $uid, $gid, $start_datetime, $end_datetime, ...$search_params);
} else {
    $stats_notes_stmt->bind_param("isss", $uid, $gid, $start_datetime, $end_datetime);
}
$stats_notes_stmt->execute();
$stats_notes_result = $stats_notes_stmt->get_result();
$stats_by_notes = [];
while ($row = $stats_notes_result->fetch_assoc()) {
    $stats_by_notes[] = $row;
}
$stats_notes_stmt->close();

// =========================
// 6. STATISTICS BY EXCHANGE RATE
// =========================
$stats_exchange_query = "
    SELECT 
        COALESCE(transaction_exchange_rate, ?) as exchange_rate,
        SUM(CASE WHEN type='IN' THEN amount ELSE 0 END) as deposit,
        SUM(CASE WHEN type='IN' THEN amount ELSE 0 END) / COALESCE(transaction_exchange_rate, ?) as conversion_u
    FROM transactions
    WHERE telegram_user_id = ?
    AND group_id = ?
    AND created_at BETWEEN ? AND ?
    $search_condition
    GROUP BY COALESCE(transaction_exchange_rate, ?)
    HAVING deposit != 0
";
$stats_exchange_stmt = $conn->prepare($stats_exchange_query);
if (!empty($search_params)) {
    $stats_exchange_stmt->bind_param("ddisssd" . $search_types, 
        $global_exchange_rate, 
        $global_exchange_rate, 
        $uid,
        $gid,
        $start_datetime, 
        $end_datetime, 
        $global_exchange_rate,
        ...$search_params
    );
} else {
    $stats_exchange_stmt->bind_param("ddisssd", 
        $global_exchange_rate, 
        $global_exchange_rate, 
        $uid,
        $gid,
        $start_datetime, 
        $end_datetime, 
        $global_exchange_rate
    );
}
$stats_exchange_stmt->execute();
$stats_exchange_result = $stats_exchange_stmt->get_result();
$stats_by_exchange = [];
while ($row = $stats_exchange_result->fetch_assoc()) {
    $stats_by_exchange[] = $row;
}
$stats_exchange_stmt->close();

// Calculate summary
$total_deposits_usdt = $total_deposits / ($global_exchange_rate > 0 ? $global_exchange_rate : 1);
$total_issued_usdt = $total_issued / ($global_exchange_rate > 0 ? $global_exchange_rate : 1);
$not_yet_distributed = $total_deposits_usdt - $total_issued_usdt;

// Previous/Next day calculation
$current_date = date('Y-m-d', strtotime($start_datetime));
$prev_date = date('Y-m-d', strtotime($current_date . ' -1 day'));
$next_date = date('Y-m-d', strtotime($current_date . ' +1 day'));

$conn->close();
?>
<!DOCTYPE html>
<html lang="zh">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>用户账本 - <?php echo htmlspecialchars($username); ?></title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css">
    <style>
        body {
            background: #f5f5f5;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Microsoft YaHei', 'SimHei', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
        }
        .container {
            max-width: 900px;
            margin: 20px auto;
            background: white;
            padding: 20px;
            border-radius: 4px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .header-section {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #e9ecef;
        }
        .date-filters {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 4px;
            margin-bottom: 20px;
        }
        .date-filters input[type="datetime-local"] {
            font-size: 14px;
        }
        .search-section {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 4px;
            margin-bottom: 20px;
        }
        .section-title {
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 10px;
            padding: 8px 12px;
            background: #e9ecef;
            border-left: 4px solid #0d6efd;
        }
        table {
            font-size: 13px;
        }
        table thead {
            background: #0d6efd;
            color: white;
        }
        table thead th {
            font-weight: 500;
            padding: 10px;
        }
        table tbody td {
            padding: 8px 10px;
            vertical-align: middle;
        }
        table tbody tr:nth-child(even) {
            background: #f8f9fa;
        }
        .stats-section {
            margin-bottom: 30px;
        }
        .bottom-summary {
            background: #f8f9fa;
            padding: 15px;
            border-radius: 4px;
            margin-top: 30px;
        }
        .bottom-summary .info-row {
            display: flex;
            justify-content: space-between;
            margin-bottom: 8px;
            font-size: 14px;
        }
        .bottom-actions {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 20px;
        }
        .btn-excel {
            background: #28a745;
            color: white;
            padding: 8px 20px;
            border: none;
            border-radius: 4px;
            text-decoration: none;
            font-size: 14px;
        }
        .btn-excel:hover {
            background: #218838;
            color: white;
        }
        .nav-buttons button, .nav-buttons a {
            margin: 0 5px;
            font-size: 14px;
        }
        .badge-count {
            background: #6c757d;
            color: white;
            padding: 2px 8px;
            border-radius: 12px;
            font-size: 11px;
            margin-left: 8px;
        }
    </style>
</head>
<body>
<div class="container">
    <!-- Header -->
    <div class="header-section">
        <div>
            <h4 class="mb-1">用户账本</h4>
            <small class="text-muted">用户: <?php echo htmlspecialchars($username); ?> (ID: <?php echo htmlspecialchars($uid); ?>)</small>
        </div>
    </div>

    <!-- Date display (read-only) -->
    <div class="date-filters">
        <div class="alert alert-info mb-3">
            <strong>当前日期:</strong> <?php echo date('Y-m-d', strtotime($start_datetime)); ?>
            <br>
            <small class="text-muted">链接仅显示当日数据，24小时后过期</small>
        </div>
    </div>

    <!-- Search Section -->
    <div class="search-section">
        <form method="GET" class="row g-2">
            <input type="hidden" name="token" value="<?php echo htmlspecialchars($token); ?>">
            <div class="col-md-6">
                <input type="text" name="search" class="form-control" 
                       placeholder="请输入您想要搜索的姓名或关键字"
                       value="<?php echo htmlspecialchars($search_query); ?>">
            </div>
            <div class="col-md-4">
                <select name="search_by" class="form-select">
                    <option value="marker" <?php echo $search_by === 'marker' ? 'selected' : ''; ?>>按标记</option>
                    <option value="operator" <?php echo $search_by === 'operator' ? 'selected' : ''; ?>>按操作员</option>
                    <option value="remark" <?php echo $search_by === 'remark' ? 'selected' : ''; ?>>按备注</option>
                </select>
            </div>
            <div class="col-md-2">
                <button type="submit" class="btn btn-primary w-100">搜索</button>
            </div>
        </form>
    </div>

    <!-- 1. Deposits Section -->
    <div class="stats-section">
        <div class="section-title">
            存款 
            <span class="badge-count"><?php echo count($deposits); ?> 条交易</span>
        </div>
        <table class="table table-bordered table-hover mb-0">
            <thead>
                <tr>
                    <th>时间</th>
                    <th>金额</th>
                    <th>标记</th>
                    <th>操作员</th>
                    <th>备注</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($deposits) === 0): ?>
                    <tr>
                        <td colspan="5" class="text-center text-muted">未找到存款记录</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($deposits as $deposit): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($deposit['created_at']); ?></td>
                            <td><?php 
                                $amount_usdt = $deposit['amount'];
                                $rate = $deposit['transaction_exchange_rate'] ?? $global_exchange_rate;
                                $amount_inr = $amount_usdt * $rate;
                                echo number_format($amount_usdt, 0) . ' U/ ' . number_format($amount_inr, 0);
                            ?></td>
                            <td><?php echo htmlspecialchars($deposit['marker']); ?></td>
                            <td><?php echo htmlspecialchars($deposit['operator_name'] ?: '-'); ?></td>
                            <td><?php echo htmlspecialchars($deposit['remark'] ?: '-'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- 2. Issued Section -->
    <div class="stats-section">
        <div class="section-title">
            已下发 
            <span class="badge-count"><?php echo count($issued); ?> 项</span>
        </div>
        <table class="table table-bordered table-hover mb-0">
            <thead>
                <tr>
                    <th>时间</th>
                    <th>金额</th>
                    <th>标记</th>
                    <th>操作员</th>
                    <th>备注</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($issued) === 0): ?>
                    <tr>
                        <td colspan="5" class="text-center text-muted">未找到下发记录</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($issued as $item): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($item['created_at']); ?></td>
                            <td><?php 
                                $amount_usdt = $item['amount'];
                                $rate = $item['transaction_exchange_rate'] ?? $global_exchange_rate;
                                $amount_inr = $amount_usdt * $rate;
                                echo number_format($amount_usdt, 0) . ' U/ ' . number_format($amount_inr, 0);
                            ?></td>
                            <td><?php echo htmlspecialchars($item['marker']); ?></td>
                            <td><?php echo htmlspecialchars($item['operator_name'] ?: '-'); ?></td>
                            <td><?php echo htmlspecialchars($item['remark'] ?: '-'); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- 3. Statistics by Marker -->
    <div class="stats-section">
        <div class="section-title">
            统计（按标记） 
            <span class="badge-count"><?php echo count($stats_by_marker); ?> 人</span>
        </div>
        <table class="table table-bordered table-hover mb-0">
            <thead>
                <tr>
                    <th>用户名</th>
                    <th>存款</th>
                    <th>已下发</th>
                    <th>未下发</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($stats_by_marker) === 0): ?>
                    <tr>
                        <td colspan="4" class="text-center text-muted">无数据</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($stats_by_marker as $stat): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($stat['username']); ?></td>
                            <td><?php echo number_format($stat['deposit'], 2); ?></td>
                            <td><?php echo number_format($stat['already_issued'], 2); ?></td>
                            <td><?php echo number_format($stat['not_issued'], 2); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- 4. Statistics by Operator -->
    <div class="stats-section">
        <div class="section-title">
            统计（按操作员） 
            <span class="badge-count"><?php echo count($stats_by_operator); ?> 人</span>
        </div>
        <table class="table table-bordered table-hover mb-0">
            <thead>
                <tr>
                    <th>用户名</th>
                    <th>存款</th>
                    <th>已下发</th>
                    <th>未下发</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($stats_by_operator) === 0): ?>
                    <tr>
                        <td colspan="4" class="text-center text-muted">无数据</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($stats_by_operator as $stat): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($stat['username']); ?></td>
                            <td><?php echo number_format($stat['deposit'], 2); ?></td>
                            <td><?php echo number_format($stat['already_issued'], 2); ?></td>
                            <td><?php echo number_format($stat['not_issued'], 2); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- 5. Statistics by Notes -->
    <div class="stats-section">
        <div class="section-title">
            统计（按备注） 
            <span class="badge-count"><?php echo count($stats_by_notes); ?> 人</span>
        </div>
        <table class="table table-bordered table-hover mb-0">
            <thead>
                <tr>
                    <th>用户名</th>
                    <th>存款</th>
                    <th>已下发</th>
                    <th>余额</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($stats_by_notes) === 0): ?>
                    <tr>
                        <td colspan="4" class="text-center text-muted">无数据</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($stats_by_notes as $stat): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($stat['username'] ?: '无备注'); ?></td>
                            <td><?php echo number_format($stat['deposit'], 2); ?></td>
                            <td><?php echo number_format($stat['already_issued'], 2); ?></td>
                            <td><?php echo number_format($stat['balance'], 2); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- 6. Statistics by Exchange Rate -->
    <div class="stats-section">
        <div class="section-title">
            统计（按汇率分类）
        </div>
        <table class="table table-bordered table-hover mb-0">
            <thead>
                <tr>
                    <th>汇率</th>
                    <th>存款</th>
                    <th>换算U</th>
                </tr>
            </thead>
            <tbody>
                <?php if (count($stats_by_exchange) === 0): ?>
                    <tr>
                        <td colspan="3" class="text-center text-muted">无数据</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($stats_by_exchange as $stat): ?>
                        <tr>
                            <td><?php echo number_format($stat['exchange_rate'], 4); ?></td>
                            <td><?php echo number_format($stat['deposit'], 2); ?></td>
                            <td><?php echo number_format($stat['conversion_u'], 2); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>

    <!-- Bottom Summary -->
    <div class="bottom-summary">
        <div class="info-row">
            <span><strong>汇率:</strong> <?php echo number_format($global_exchange_rate, 4); ?></span>
        </div>
        <div class="info-row">
            <span><strong>费率:</strong> <?php echo number_format($global_fee_rate, 2); ?>%</span>
        </div>
        <div class="info-row">
            <span><strong>总存款:</strong> <?php echo number_format($total_deposits, 0); ?> ( USDT )</span>
            <span><?php echo number_format($total_deposits_usdt, 0); ?> 0 ( USDT )</span>
        </div>
        <div class="info-row">
            <span><strong>0 USDT 应下发。</strong></span>
        </div>
        <div class="info-row">
            <span><strong>已下发:</strong> <?php echo number_format($total_issued, 0); ?> ( USDT )</span>
            <span><?php echo number_format($total_issued_usdt, 0); ?> 0 ( USDT )</span>
        </div>
        <div class="info-row">
            <span><strong>尚未分配:</strong> <?php echo number_format($total_deposits - $total_issued, 1); ?> ( USDT )</span>
            <span><?php echo number_format($not_yet_distributed, 1); ?> 0 ( USDT )</span>
        </div>
    </div>

    <!-- Bottom Actions -->
    <div class="bottom-actions">
        <div>
            <small class="text-muted">链接有效期至: <?php echo date('Y-m-d H:i', $payload['created'] + (24 * 3600)); ?></small>
        </div>
        <div>
            <a href="export.php?token=<?php echo urlencode($token); ?>" 
               class="btn-excel">下载Excel</a>
        </div>
    </div>
</div>

</body>
</html>