<?php
// file: link_crypto.php - Secure link generation and decryption
require_once "config.php";

/**
 * Generate a secure encrypted token for user ledger links
 * 
 * @param int $uid Telegram user ID
 * @param string $start_date Start date in 'Y-m-d H:i:s' format
 * @param string $end_date End date in 'Y-m-d H:i:s' format
 * @return string Base64-encoded encrypted token
 */
function generate_secure_token($uid, $gid, $start_date, $end_date) {
    // Create payload with timestamp
    $payload = [
        'uid' => $uid,
        'gid' => $gid,
        'start_date' => $start_date,
        'end_date' => $end_date,
        'created' => time()
    ];
    
    $json = json_encode($payload);
    
    // Encryption settings
    $method = 'AES-256-CBC';
    $key = hash('sha256', API_SECRET, true);
    $iv = openssl_random_pseudo_bytes(16);
    
    // Encrypt - use OPENSSL_RAW_DATA to match decryption
    $encrypted = openssl_encrypt($json, $method, $key, OPENSSL_RAW_DATA, $iv);
    
    // Combine IV + encrypted data
    $result = base64_encode($iv . $encrypted);
    
    return $result;
}

/**
 * Decrypt and validate a secure token
 * 
 * @param string $encrypted_token Base64-encoded encrypted token
 * @return array|null Decrypted payload array or null if invalid
 */
function decrypt_secure_token($encrypted_token) {
    if (empty($encrypted_token)) {
        return null;
    }
    
    try {
        $method = 'AES-256-CBC';
        $key = hash('sha256', API_SECRET, true);
        
        // Convert URL-safe base64 to standard base64
        // Python uses urlsafe_b64encode which replaces + with - and / with _
        $encrypted_token = str_replace(['-', '_'], ['+', '/'], $encrypted_token);
        
        // Decode
        $decoded = base64_decode($encrypted_token, true);
        if ($decoded === false || strlen($decoded) < 16) {
            error_log("Token decode failed or too short");
            return null;
        }
        
        // Extract IV (first 16 bytes) and ciphertext
        $iv = substr($decoded, 0, 16);
        $ciphertext = substr($decoded, 16);
        
        // Decrypt (use OPENSSL_RAW_DATA because Python returns raw bytes)
        $decrypted = openssl_decrypt($ciphertext, $method, $key, OPENSSL_RAW_DATA, $iv);
        
        if ($decrypted === false) {
            error_log("Token decryption failed");
            return null;
        }
        
        // Parse JSON
        $payload = json_decode($decrypted, true);
        
        if (!$payload || !isset($payload['uid']) || !isset($payload['created'])) {
            error_log("Token payload invalid");
            return null;
        }
        
        return $payload;
    } catch (Exception $e) {
        error_log("Token decryption error: " . $e->getMessage());
        return null;
    }
}

/**
 * Check if a token has expired
 * 
 * @param int $created_timestamp Unix timestamp when token was created
 * @param int $expiry_hours Number of hours until expiration (default: 24)
 * @return bool True if expired, false otherwise
 */
function is_token_expired($created_timestamp, $expiry_hours = 24) {
    $current_time = time();
    $expiry_time = $created_timestamp + ($expiry_hours * 3600);
    
    return $current_time > $expiry_time;
}

/**
 * Check if a date is from the current day only
 * 
 * @param string $date_string Date string in 'Y-m-d H:i:s' format
 * @return bool True if date is from today, false otherwise
 */
function is_current_day($date_string) {
    $today = date('Y-m-d');
    $check_date = date('Y-m-d', strtotime($date_string));
    
    return $today === $check_date;
}
?>
