from telegram.ext import Updater, CommandHandler, MessageHandler, Filters
import requests
import logging
import re
from datetime import datetime

# =============== CONFIG ==================
BOT_TOKEN = "8506512760:AAF59feSSTx5YDdnVlEIPeEkMKNqUT3jDco"

# Production server URLs
ADD_TX_URL = "https://partylimes.com/ledger/api/add_transaction.php"
SET_RATE_URL = "https://partylimes.com/ledger/api/set_exchange.php"
GET_SUMMARY_URL = "https://partylimes.com/ledger/api/get_summary.php"
GET_LOGS_URL = "https://partylimes.com/ledger/api/get_logs.php"

API_SECRET = "test_ledger_9f8e7d6c5b4a"

# public per-user page
PUBLIC_LEDGER_BASE = "https://partylimes.com/ledger/user_ledger.php"

# =============== LOGGING ==================
logging.basicConfig(
    format="%(asctime)s - %(name)s - %(levelname)s - %(message)s",
    level=logging.INFO,
)
logger = logging.getLogger(__name__)


# =============== HELPERS ==================
def get_exchange_rate(group_id):
    """
    Fetches the exchange_rate for a specific group_id from the API.
    """
    # 1. Validation for group_id
    if not group_id:
        logger.error("Get Rate Error: group_id is missing or null.")
        return None

    try:
        # 2. Sending GET request with parameters
        params = {"group_id": group_id}
        response = requests.get(GET_GROUP_RATE_URL, params=params, timeout=5)
        
        # 3. Handle HTTP errors (e.g., 404, 500)
        response.raise_for_status()
        
        data = response.json()

        # 4. Check if the API returned an error message or valid data
        if isinstance(data, dict) and "error" in data:
            logger.warning(f"API returned error for group {group_id}: {data['error']}")
            return None
        
        # Extract the specific rate (assumes API returns {"exchange_rate": value, ...})
        rate = data.get("exchange_rate")
        
        if rate is not None:
            logger.info(f"Successfully retrieved rate {rate} for group {group_id}")
            return rate
        else:
            logger.error(f"Rate field missing in API response for group {group_id}")
            return None

    except requests.exceptions.RequestException as e:
        logger.error(f"Network error while fetching rate for group {group_id}: {str(e)}")
        return None
    except ValueError:
        logger.error(f"Invalid JSON response from API for group {group_id}")
        return None


def generate_secure_link(uid, start_date, end_date):
    """
    Generate a secure encrypted link for user ledger using the API.
    """
    try:
        # URL encode the dates
        import urllib.parse
        params = {
            "uid": uid,
            "start_date": start_date,
            "end_date": end_date
        }
        
        # Call the generate_link API
        generate_link_url = f"{LOCAL_BASE}/api/generate_link.php"
        response = requests.get(generate_link_url, params=params, timeout=5)
        
        if response.status_code == 200:
            data = response.json()
            return data.get("link", "")
        else:
            logger.error(f"Failed to generate secure link: {response.status_code}")
            return ""
    except Exception as e:
        logger.error(f"Error generating secure link: {str(e)}")
        return ""




def parse_amount_and_remark(args):
    if not args:
        return None, None
    try:
        amount = float(args[0])
    except ValueError:
        return None, None
    remark = " ".join(args[1:]) if len(args) > 1 else ""
    return amount, remark


def send_tx_to_api(payload):
    try:
        r = requests.post(ADD_TX_URL, data=payload, timeout=5)
        logger.info(f"Add TX API Response: Status={r.status_code}, Body={r.text}")
        if r.status_code == 200:
            return True, r.text
        return False, f"HTTP {r.status_code}: {r.text}"
    except Exception as e:
        logger.error(f"Add TX API Exception: {str(e)}")
        return False, str(e)


def send_rate_to_api(rate_value, group_id, group_name):
    try:
        r = requests.post(
            SET_RATE_URL,
            data={"api_secret": API_SECRET, 
                "exchange_rate": rate_value, 
                "group_id": group_id,
                "group_name": group_name},
            timeout=5,
        )
        logger.info(f"Set Rate API Response: Status={r.status_code}, Body={r.text}")
        if r.status_code == 200:
            return True, r.text
        return False, f"HTTP {r.status_code}: {r.text}"
    except Exception as e:
        logger.error(f"Set Rate API Exception: {str(e)}")
        return False, str(e)


def is_admin(update, context):
    chat = update.effective_chat
    user = update.effective_user
    try:
        member = context.bot.get_chat_member(chat.id, user.id)
        logger.info(
            "Chat %s: user %s status = %s",
            chat.id,
            user.username or user.id,
            member.status,
        )
        return member.status in ("administrator", "creator")
    except Exception as e:
        logger.error("get_chat_member failed: %s", e)
        return False


def compute_daily_stats(rate, telegram_user_id=None, include_rows=False):
    """
    Per-user daily stats (for one Telegram user).
    Returns:
      {
        count_in, count_out,
        total_in_u, total_out_u,
        total_in_inr, total_out_inr,
        should_inr, should_u,
        issued_inr, issued_u,
        balance_inr, balance_u,
        last_in, last_out (if include_rows=True)
      }, today_date
    """
    from datetime import datetime, timedelta
    
    # Get today's date in local timezone
    today = datetime.now().strftime("%Y-%m-%d")
    
    # Also query yesterday to account for timezone differences
    yesterday = (datetime.now() - timedelta(days=1)).strftime("%Y-%m-%d")
    
    try:
        # Query for both today and yesterday to catch timezone issues
        r_today = requests.get(GET_LOGS_URL, params={"date": today}, timeout=5)
        r_yesterday = requests.get(GET_LOGS_URL, params={"date": yesterday}, timeout=5)
        
        rows_today = r_today.json() if r_today.status_code == 200 else []
        rows_yesterday = r_yesterday.json() if r_yesterday.status_code == 200 else []
        
        # Combine and get unique rows
        rows = rows_today + rows_yesterday
        logger.info(f"get_logs API returned {len(rows_today)} rows for {today}, {len(rows_yesterday)} rows for {yesterday}")
    except Exception as e:
        logger.error("get_logs error: %s", e)
        return None, today

    # Filter rows for a specific Telegram user, if given
    if telegram_user_id is not None:
        tid = str(telegram_user_id)
        rows = [row for row in rows if str(row.get("telegram_user_id")) == tid]
        logger.info(f"After filtering for user {tid}: {len(rows)} rows")

    rate = float(rate or 0)

    total_in_u = sum(float(row.get("amount", 0) or 0) for row in rows if row.get("type") == "IN")
    total_out_u = sum(float(row.get("amount", 0) or 0) for row in rows if row.get("type") == "OUT")
    count_in = sum(1 for row in rows if row.get("type") == "IN")
    count_out = sum(1 for row in rows if row.get("type") == "OUT")

    total_in_inr = total_in_u * rate
    total_out_inr = total_out_u * rate

    should_inr = total_in_inr      # 应下发 (fiat)
    should_u   = total_in_u        # 应下发 (USDT)

    issued_inr = total_out_inr     # 已下发 (fiat)
    issued_u   = total_out_u       # 已下发 (USDT)

    balance_inr = should_inr - issued_inr
    balance_u   = should_u - issued_u

    result = {
        "count_in": count_in,
        "count_out": count_out,
        "total_in_u": total_in_u,
        "total_out_u": total_out_u,
        "total_in_inr": total_in_inr,
        "total_out_inr": total_out_inr,
        "should_inr": should_inr,
        "should_u": should_u,
        "issued_inr": issued_inr,
        "issued_u": issued_u,
        "balance_inr": balance_inr,
        "balance_u": balance_u,
    }

    if include_rows:
        last_in = next((row for row in reversed(rows) if row.get("type") == "IN"), None)
        last_out = next((row for row in reversed(rows) if row.get("type") == "OUT"), None)
        result["last_in"] = last_in
        result["last_out"] = last_out

    return result, today


# =============== COMMANDS ==================
def start(update, context):
    msg = (
        "👋 Ledger Bot ready (LOCAL MODE).\n\n"
        "Slash commands:\n"
        "• /in <amount> <remark>  → deposit (IN)\n"
        "• /out <amount> <remark> → payout (OUT)\n"
        "• /today                → summary for today\n\n"
        "Text flow in group:\n"
        "• Admin: 'set rate 108'\n"
        "• Customer sends screenshot\n"
        "• Admin: 'in 80' to record deposit\n"
        "• Admin: '500u' or 'out 500u' to record payout + summary\n"
        "• Admin: '100000i' or 'out 100000i' to record INR payout (NEW)"
    )
    update.message.reply_text(msg)


def handle_transaction(update, context, tx_type):
    user = update.effective_user
    chat = update.effective_chat

    amount, remark = parse_amount_and_remark(context.args)
    if amount is None:
        update.message.reply_text(
            f"❌ Invalid usage.\n\nFormat:\n/{tx_type.lower()} <amount> <remark>\n"
            f"Example:\n/{tx_type.lower()} 100 test entry"
        )
        return

    payload = {
        "api_secret": API_SECRET,
        "telegram_user_id": user.id,
        "telegram_username": user.username or user.first_name or "",
        "type": tx_type,
        "amount": amount,
        "remark": remark,
        "group_id": chat.id,
        "operator_name": user.username or user.first_name or "",
        "currency": "USDT",
    }

    ok, info = send_tx_to_api(payload)
    if ok:
        update.message.reply_text(
            f"✅ {tx_type} entry saved: {amount}\nRemark: {remark or '—'}"
        )
    else:
        logger.error("API error: %s", info)
        update.message.reply_text("⚠️ Failed to save to server. Please try again later.")


def in_command(update, context):
    handle_transaction(update, context, tx_type="IN")


def out_command(update, context):
    handle_transaction(update, context, tx_type="OUT")


def today_command(update, context):
    today = datetime.now().strftime("%Y-%m-%d")
    try:
        r = requests.get(GET_SUMMARY_URL, params={"date": today}, timeout=5)
        data = r.json()
        total_in = float(data.get("IN", 0) or 0)
        total_out = float(data.get("OUT", 0) or 0)
        net = total_in - total_out
        msg = (
            f"📅 Summary for {today}\n\n"
            f"💰 IN : {total_in}\n"
            f"💸 OUT: {total_out}\n"
            f"➡️ Net: {net}"
        )
        update.message.reply_text(msg)
    except Exception as e:
        logger.error("Error fetching summary: %s", e)
        update.message.reply_text("⚠️ Could not fetch summary from server.")


# =============== GROUP FLOW ==================
def handle_screenshot(update, context):
    """Store last customer who sent a photo in this chat."""
    user = update.effective_user
    if user.is_bot:
        return
    chat = update.effective_chat
    context.chat_data["last_customer"] = {
        "id": user.id,
        "username": user.username or "",
        "name": user.full_name,
    }
    logger.info("Stored last_customer in chat %s: %s", chat.id, context.chat_data["last_customer"])


def admin_text_handler(update, context):
    """
    - 'set rate 108' / 'set exchange rate 108' -> set rate (admin only)
    - 'in 80', 'in 80 test'                    -> IN entry + summary + link
    - payout text '500u', 'out 500u', etc.    -> OUT entry + full summary + link
    - payout INR '100000i', 'out 100000i'     -> OUT entry (INR converted) + summary
    """
    message = update.message
    text = (message.text or "").strip()
    chat = update.effective_chat
    user = update.effective_user

    logger.info("Text in chat %s from %s: %r", chat.id, user.username or user.id, text)

    if chat.type not in ("group", "supergroup"):
        return

    # ---------- 1) SET RATE ----------
    rate_match = re.match(r"(?i)^set\s*(exchange\s*)?rate\s+(\d+)\s*$", text)
    if rate_match:
        if not is_admin(update, context):
            message.reply_text("⚠️ Only group admins can set exchange rate.")
            return

        rate_value = int(rate_match.group(2))
        group_id = chat.id
        group_name = chat.title

        ok, info = send_rate_to_api(rate_value, group_id, group_name)
        if ok:
            context.chat_data["current_rate"] = float(rate_value)
            message.reply_text(
                f"✅ Exchange rate set successfully.\nCurrent rate: {rate_value}"
            )
            logger.info("Rate %s set in chat %s by %s", rate_value, chat.id, user.username)
        else:
            logger.error("Rate API error: %s", info)
            message.reply_text("⚠️ Failed to update rate on server.")
        return

    # ---------- 2) IN TEXT (DEPOSIT) ----------
    dep_match = re.match(r"(?i)^in\s+(\d+(?:\.\d+)?)\s*(.*)$", text)
    if dep_match:
        if not is_admin(update, context):
            return

        # Check if customer screenshot exists
        customer = context.chat_data.get("last_customer")
        if not customer:
            message.reply_text(
                "⚠️ No customer screenshot found.\nAsk the customer to send a screenshot first."
            )
            return

        amount_usdt = float(dep_match.group(1))
        remark = dep_match.group(2).strip()
        group_id = chat.id
        exchange_rate = float(get_exchange_rate(group_id) or 0.00)
        amount_in_rs = exchange_rate * amount_usdt

        payload = {
            "api_secret": API_SECRET,
            "telegram_user_id": customer["id"],
            "telegram_username": customer["username"] or customer["name"],
            "type": "IN",
            "amount": amount_usdt,
            "amount_in_rs": amount_in_rs,
            "remark": remark or text,
            "group_id": group_id,
            "operator_name": user.username or user.first_name or "",
            "currency": "USDT",
        }

        ok, info = send_tx_to_api(payload)
        if not ok:
            logger.error("API error on IN via text: %s", info)
            message.reply_text("⚠️ Failed to save IN entry to server.")
            return


        # Full summary per user (matching OUT format)
        rate = float(get_exchange_rate(group_id) or 0.00)
        stats, today = compute_daily_stats(rate, customer["id"])
        now_time = datetime.now().strftime("%H:%M:%S")
        cust_label = customer["username"] or customer["name"]

        if stats:
            deposits = stats["count_in"]
            withdrawals = stats["count_out"]
            total_in_inr = stats["total_in_inr"]
            total_in_u = stats["total_in_u"]
            should_inr = stats["should_inr"]
            should_u = stats["should_u"]
            issued_inr = stats["issued_inr"]
            issued_u = stats["total_out_u"]
            balance_inr = stats["balance_inr"]
            balance_u = stats["balance_u"]
            fee_rate = context.chat_data.get("fee_rate", 0)

            amount_inr = amount_usdt * float(rate) if rate else 0
            link = generate_secure_link(customer['id'], f"{today} 00:00:00", f"{today} 23:59:59")

            reply_text = (
                f"今日入款（{deposits}笔）\n\n"
                f"今日下发（{withdrawals}笔）\n"
                f"{now_time}  {amount_inr:.0f} / {rate:.0f}={amount_usdt:.1f}U {cust_label}\n\n"
                f"总入款：{total_in_inr:.0f} ({total_in_u:.0f}U)\n"
                f"汇率：{rate:.0f}\n"
                f"交易费率：{fee_rate:.0f}%\n\n"
                f"应下发：{should_inr:.0f}  |  {should_u:.0f}U\n"
                f"已下发：{issued_inr:.0f}  |  {issued_u:.0f}U\n"
                f"余额：{balance_inr:.0f}  |  {balance_u:.0f}U\n\n"
                f"🔗 完整账单: {link}"
            )
            message.reply_text(reply_text)
            return
        else:
            # Fallback if no stats available
            deposits = 1
            amount_inr = amount_usdt * float(rate) if rate else 0
            link = generate_secure_link(customer['id'], f"{today} 00:00:00", f"{today} 23:59:59")

            reply_text = (
                f"今日入款（{deposits}笔）\n"
                f"{now_time}  {amount_inr:.0f} / {rate:.0f}={amount_usdt:.1f}U {cust_label}\n\n"
                f"🔗 完整账单: {link}"
            )

        message.reply_text(reply_text)
        return

    # ---------- 3) OUT TEXT (PAYOUT) - SUPPORTS BOTH USDT AND INR ----------
    if not is_admin(update, context):
        return

    # Updated regex to detect both 'u' (USDT) and 'i' (INR)
    amt_match = re.search(r"(\d+)\s*([ui])\b", text, flags=re.IGNORECASE)
    if not amt_match:
        return

    amount_value = float(amt_match.group(1))
    currency_suffix = amt_match.group(2).lower()  # 'u' or 'i'

    group_id = chat.id
    rate = float(get_exchange_rate(group_id) or 0.00)
    customer = context.chat_data.get("last_customer")

    # Convert INR to USDT if needed
    if currency_suffix == 'i':
        # Input is in INR, convert to USDT
        if not rate or rate <= 0:
            message.reply_text("⚠️ Cannot process INR amount. Please set the exchange rate first (e.g. 'set rate 108').")
            return
        amount_usdt = amount_value / rate
        amount_inr = amount_value
    else:
        # Input is in USDT
        amount_usdt = amount_value
        amount_inr = amount_usdt * rate

    if not customer:
        message.reply_text(
            "⚠️ No customer screenshot found.\nAsk the customer to send a screenshot first."
        )
        return

    payload = {
        "api_secret": API_SECRET,
        "telegram_user_id": customer["id"],
        "telegram_username": customer["username"] or customer["name"],
        "type": "OUT",
        "amount": amount_usdt,
        "amount_in_rs": amount_inr,
        "remark": text,
        "group_id": chat.id,
        "operator_name": user.username or user.first_name or "",
        "currency": "USDT",
    }

    ok, info = send_tx_to_api(payload)
    if not ok:
        logger.error("API error on OUT via text: %s", info)
        message.reply_text("⚠️ Failed to save payout to server.")
        return

    cust_label = customer["username"] or customer["name"]
    now_time = datetime.now().strftime("%H:%M:%S")

    stats, today = compute_daily_stats(rate, customer["id"])
    link = generate_secure_link(customer['id'], f"{today} 00:00:00", f"{today} 23:59:59")

    if stats:
        deposits = stats["count_in"]
        withdrawals = stats["count_out"]
        total_in_inr = stats["total_in_inr"]
        total_in_u = stats["total_in_u"]
        should_inr = stats["should_inr"]
        should_u = stats["should_u"]
        issued_inr = stats["issued_inr"]
        issued_u = stats["total_out_u"]
        balance_inr = stats["balance_inr"]
        balance_u = stats["balance_u"]
        fee_rate = context.chat_data.get("fee_rate", 0)  # Get fee rate if stored

        # Build withdrawal message matching reference format
        if currency_suffix == 'i':
            # User entered INR
            reply_text = (
                f"今日入款（{deposits}笔）\n\n"
                f"今日下发（{withdrawals}笔）\n"
                f"{now_time}  {amount_inr:.0f}i → {amount_usdt:.2f}U {cust_label}\n\n"
            )
        else:
            # User entered USDT
            reply_text = (
                f"今日入款（{deposits}笔）\n\n"
                f"今日下发（{withdrawals}笔）\n"
                f"{now_time}  {amount_usdt:.0f}U {cust_label}\n\n"
            )
        
        reply_text += (
            f"总入款：{total_in_inr:.0f} ({total_in_u:.0f}U)\n"
            f"汇率：{rate:.0f}\n"
            f"交易费率：{fee_rate:.0f}%\n\n"
            f"应下发：{should_inr:.0f}  |  {should_u:.0f}U\n"
            f"已下发：{issued_inr:.0f}  |  {issued_u:.0f}U\n"
            f"余额：{balance_inr:.0f}  |  {balance_u:.0f}U\n\n"
            f"🔗 完整账单: {link}"
        )
        message.reply_text(reply_text)
    else:
        amount_inr = amount_usdt * float(rate)
        reply_text = (
            f"✅ Recorded payout: {amount_usdt:.2f}U to {cust_label}\n"
            f"Rate: {rate} → Approx: {amount_inr:.2f}\n\n"
            f"🔗 Full bill: {link}"
        )
        message.reply_text(reply_text)


def error_handler(update, context):
    logger.error("Exception in ledger bot:", exc_info=context.error)


# =============== MAIN ==================
def main():
    updater = Updater(BOT_TOKEN, use_context=True)
    dp = updater.dispatcher

    dp.add_handler(CommandHandler("start", start))
    dp.add_handler(CommandHandler("help", start))
    dp.add_handler(CommandHandler("in", in_command))
    dp.add_handler(CommandHandler("out", out_command))
    dp.add_handler(CommandHandler("today", today_command))

    dp.add_handler(MessageHandler(Filters.photo, handle_screenshot))
    dp.add_handler(MessageHandler(Filters.text & ~Filters.command, admin_text_handler))

    dp.add_error_handler(error_handler)

    updater.start_polling()
    logger.info("🚀 Ledger bot started in LOCAL MODE...")
    logger.info(f"📍 Connecting to: {LOCAL_BASE}")
    updater.idle()


if __name__ == "__main__":
    main()
