<?php
/**
 * Currency Helper for INR/USDT Detection
 * Parses amount strings with 'i' (INR) or 'u' (USDT) suffixes
 */

/**
 * Parse amount with currency detection
 * 
 * Examples:
 *   "100" or "100u" → USDT
 *   "8500i" → INR
 * 
 * @param string $amount_str The amount string (e.g., "100", "100u", "8500i")
 * @param float $exchange_rate Current exchange rate (1 USDT = X INR)
 * @return array|null Array with ['amount_usdt', 'amount_inr', 'currency'] or null if invalid
 */
function parse_amount_with_currency($amount_str, $exchange_rate) {
    if (empty($amount_str) || !$exchange_rate || $exchange_rate <= 0) {
        return null;
    }
    
    $amount_str = trim($amount_str);
    
    // Check for INR suffix (i)
    if (preg_match('/^([\d.]+)i$/i', $amount_str, $matches)) {
        $amount_inr = floatval($matches[1]);
        $amount_usdt = $amount_inr / $exchange_rate;
        
        return [
            'amount_usdt' => $amount_usdt,
            'amount_inr' => $amount_inr,
            'currency' => 'INR',
            'input_value' => $amount_inr
        ];
    }
    
    // Check for USDT suffix (u) or no suffix (default to USDT)
    if (preg_match('/^([\d.]+)u?$/i', $amount_str, $matches)) {
        $amount_usdt = floatval($matches[1]);
        $amount_inr = $amount_usdt * $exchange_rate;
        
        return [
            'amount_usdt' => $amount_usdt,
            'amount_inr' => $amount_inr,
            'currency' => 'USDT',
            'input_value' => $amount_usdt
        ];
    }
    
    return null;
}

/**
 * Get current exchange rate from settings
 * 
 * @param mysqli $conn Database connection
 * @return float Exchange rate or 0.0 if not found
 */
function get_exchange_rate_from_settings($conn) {
    $stmt = $conn->prepare("SELECT exchange_rate FROM settings WHERE id = 1");
    if ($stmt && $stmt->execute()) {
        $stmt->bind_result($exchange_rate);
        $stmt->fetch();
        $stmt->close();
        return floatval($exchange_rate ?: 0.0);
    }
    return 0.0;
}

/**
 * Example usage function
 */
function example_usage() {
    // Example with rate of 85 (1 USDT = 85 INR)
    $exchange_rate = 85;
    
    // Test USDT inputs
    $result1 = parse_amount_with_currency("100", $exchange_rate);
    // Returns: ['amount_usdt' => 100, 'amount_inr' => 8500, 'currency' => 'USDT']
    
    $result2 = parse_amount_with_currency("100u", $exchange_rate);
    // Returns: ['amount_usdt' => 100, 'amount_inr' => 8500, 'currency' => 'USDT']
    
    // Test INR input
    $result3 = parse_amount_with_currency("8500i", $exchange_rate);
    // Returns: ['amount_usdt' => 100, 'amount_inr' => 8500, 'currency' => 'INR']
    
    return [
        'example_1' => $result1,
        'example_2' => $result2,
        'example_3' => $result3
    ];
}
