<?php
require_once "../config.php";

header('Content-Type: application/json');

// Enable error logging
error_log("=== add_transaction.php called ===");
error_log("REQUEST_METHOD: " . $_SERVER['REQUEST_METHOD']);

// Allow both normal form POST and JSON body
$data = [];

if (!empty($_POST)) {
    $data = $_POST;
    error_log("Using POST data");
} else {
    $raw = file_get_contents("php://input");
    if (!empty($raw)) {
        $json = json_decode($raw, true);
        if (is_array($json)) {
            $data = $json;
            error_log("Using JSON data");
        }
    }
}

error_log("Parsed data: " . print_r($data, true));

// 1) Method check
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    http_response_code(405);
    error_log("ERROR: Invalid Request Method");
    echo json_encode(["error" => "Invalid Request Method"]);
    exit;
}

// 2) API secret check
$client_secret = $data["api_secret"] ?? "";
if ($client_secret !== API_SECRET) {
    http_response_code(401);
    error_log("ERROR: Unauthorized - secret mismatch");
    echo json_encode(["error" => "Unauthorized (invalid api_secret)"]);
    exit;
}

// 3) Read data
$telegram_user_id   = isset($data["telegram_user_id"]) ? (int)$data["telegram_user_id"] : 0;
$telegram_username  = $data["telegram_username"] ?? "";
$type               = $data["type"] ?? "";
$amount             = $data["amount"] ?? 0;
$amount_in_rs       = $data["amount_in_rs"] ?? 0;
$remark             = $data["remark"] ?? "";
$group_id           = $data["group_id"] ?? null;
$operator_name      = $data["operator_name"] ?? "";
$currency           = $data["currency"] ?? "USDT";

// Basic validation
if (!$telegram_user_id || !in_array($type, ["IN","OUT"], true) || !is_numeric($amount)) {
    http_response_code(400);
    echo json_encode(["error" => "Missing or invalid fields"]);
    exit;
}

// 5) Get current exchange rate from settings to store with transaction
$current_exchange_rate = 1.0; // default fallback
$rate_stmt = $conn->prepare("SELECT exchange_rate FROM settings WHERE group_id = ?");
if ($rate_stmt) {
    $rate_stmt->bind_param("s", $group_id);
    if ($rate_stmt->execute()) {
        $rate_stmt->bind_result($current_exchange_rate);
        $rate_stmt->fetch();
        $rate_stmt->close();
    }
}


// 4) Insert into DB
$stmt = $conn->prepare("
    INSERT INTO transactions 
    (telegram_user_id, telegram_username, type, amount, amount_in_rs, transaction_exchange_rate, remark, group_id, operator_name, currency, created_at)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
");

$stmt->bind_param(
    "issdddsiss",
    $telegram_user_id,
    $telegram_username,
    $type,
    $amount,
    $amount_in_rs,
    $current_exchange_rate,
    $remark,
    $group_id,
    $operator_name,
    $currency
);

if ($stmt->execute()) {
    echo json_encode(["status" => "success"]);
} else {
    http_response_code(500);
    echo json_encode(["error" => "DB error: " . $stmt->error]);
}

$stmt->close();
$conn->close();
?>
